<?php

namespace MartianooCore\Api;


/**
 * This class contains utility functions.
 * @package MartianooCore\Api
 */
class Utils
{
    /**
     * This function validates a date in the standard format year-month-day.
     * Returns true if date is valid and false otherwise.
     * @param string $date the date to validate
     * @return bool
     */
    public static function validateDate($date)
    {
        if (! is_string($date)) {
            return false;
        }

        $regExp = "/[2-9][0-9]{3}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])/";
        $regExp2 = "/[^0-9\\-]/";

        if (! preg_match($regExp, $date)) {
            return false;
        }

        if (preg_match($regExp2, $date)) {
            return false;
        }

        $partsOfDate = explode("-", $date);
        $year = $partsOfDate[0];
        $month = $partsOfDate[1];
        $day = $partsOfDate[2];

        return checkdate((integer) $month, (integer) $day, (integer) $year);
    }

    /**
     * This function validates a date in the standard format **year-month-day hours:minutes:secs**
     * Returns true if date is valid and false otherwise.
     * @param string $datetime the date to validate
     * @return bool
     */
    public static function validateDateTime($datetime)
    {
        if (! is_string($datetime)) {
            return false;
        }

        $regExp = "/[2-9][0-9]{3}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])\\s[0-1][0-9]|2[0-3]:[0-5][0-9]:[0-5][0-9]/";
        $regExp2 = "/[^0-9\\-\\s\\:]/";

        if (! preg_match($regExp, $datetime, $m)) {
            return false;
        }

        if (preg_match($regExp2, $datetime)) {
            return false;
        }

        $dateRegExp = "/[2-9][0-9]{3}-(0[1-9]|1[0-2])-(0[1-9]|[1-2][0-9]|3[0-1])/";
        preg_match($dateRegExp, $datetime, $matches);
        $matchedDate = $matches[0];

        $partsOfDate = explode("-", $matchedDate);
        $year = $partsOfDate[0];
        $month = $partsOfDate[1];
        $day = $partsOfDate[2];

        return checkdate((integer) $month, (integer) $day, (integer) $year);
    }

    /**
     * Returns the current date in standard format i.e. year-month-day.
     * @param integer $timestamp The optional default timestamp used.
     * @return string
     */
    public static function dateOfDay($timestamp = 0)
    {
        if (is_int($timestamp) && $timestamp > 0) {
            $date = getdate($timestamp);
        } else {
            $date = getdate();
        }

        $day = ($date['mday'] >= 10) ? $date['mday'] : '0' . $date['mday'];
        $month = ($date['mon'] >= 10) ? $date['mon'] : '0' . $date['mon'];
        $today = $date['year'] . '-' . $month. '-' . $day;

        return $today;
    }

    /**
     * Returns the current time in hours:minutes:seconds
     * @param integer $timestamp The optional default timestamp used.
     * @return string
     */
    public static function currentTime($timestamp = 0)
    {
        if (is_int($timestamp) && $timestamp > 0) {
            $date = getdate($timestamp);
        } else {
            $date = getdate();
        }

        $minutes = ($date['minutes'] >= 10) ? $date['minutes'] : '0' . $date['minutes'];
        $hours = ($date['hours'] >= 10) ? $date['hours'] : '0' . $date['hours'];
        $seconds = ($date['seconds'] >= 10) ? $date['seconds'] : '0' . $date['seconds'];
        $time = $hours . ':' . $minutes . ':' . $seconds;

        return $time;
    }

    public static function currentDateTime()
    {

        return self::dateOfDay() . ' ' . self::currentTime();
    }

    /**
     * This function validates a date in the format day/month/year.
     * Returns true if date is valid and false otherwise.
     * @param string $date the date to validate
     * @return bool
     */
    public static function dateIsCorrect($date)
    {

        $separator = '/';

        $parts = explode($separator, $date);
        if(!(count($parts) == 3)) {
            return false;
        }


        $day = (int)self::removeSpace($parts[0]);
        $month = (int)self::removeSpace($parts[1]);
        $year = (int)self::removeSpace($parts[2]);

        if(! checkdate($month, $day, $year)) {
            return false;
        }

        return true;
    }

    /**
     * Converts a date from the format day/month/year to the standard format year-month-day.
     * Returns the converted date; otherwise it returns false.
     * @param $date
     * @return mixed
     */
    public static function standardizeDate($date)
    {
        $separator = '/';

        $parts = explode($separator, $date);
        if(! (count($parts) == 3)){
            return false;
        }

        $day = (int)self::removeSpace($parts[0]);
        $month = (int)self::removeSpace($parts[1]);
        $year = (int)self::removeSpace($parts[2]);

        if(! checkdate($month, $day, $year)) {
            return false;
        }

        $finalDate = self::removeSpace($parts[2]) . '-' .
            self::removeSpace($parts[1]) . '-' .
            self::removeSpace($parts[0]);

        return $finalDate;
    }

    /**
     * @param int $min
     * @param int $max
     * @param bool $hasSymbol
     * @return string
     */
    public static function generateCode($min = 10, $max = 50, $hasSymbol = false)
    {
        $length = rand($min, $max);
        $letters = ['a','b','c','d','e','f',
            'g','h','i','j','k','l',
            'm','n','o','p','q','r',
            's','t','u','v','w','x',
            'y','z'];
        $symbols = ['_', '!', '@'];
        $code = '';

        for ($i = 1; $i <= $length; $i++){
            if($hasSymbol){
                $randKey = rand(0, 2);
                $code .= $symbols[$randKey];
            }

            $way = rand(1,2);
            if($way == 1){
                $randKey = rand(0, 25);
                $case = rand(1,2);

                if($case == 1){
                    $code .= $letters[$randKey];
                }else{
                    $code .= ucfirst($letters[$randKey]);
                }
            }else{
                $randDigit = rand(0, 9);
                $code .= $randDigit;
            }
        }

        return $code;
    }

    /*public static function writeToLog($text, $file, $line)
    {
        $date =  self::currentDateTime();
        $logFilePath = WP_PLUGIN_DIR . '/affiliatere2/logs/log.text';

        if(!file_exists($logFilePath)){
            file_put_contents($logFilePath, '');
        }

        if(file_exists($logFilePath)){
            $msg = '[' . $date . '] ';
            $msg .= $text . ' ' . $file . ':' . $line;
            $fileContent = file_get_contents($logFilePath, false);
            if(strlen($fileContent) == 0){
                $fileContent = $msg;
            }else{
                $fileContent = $fileContent . PHP_EOL . $msg;
            }

            file_put_contents($logFilePath, $fileContent);
        }
    }*/

    /**
     * @
     * @param $string
     * @return mixed
     */
    private static function removeSpace($string)
    {
        return preg_replace('/[\s]/', '', $string);
    }
}