<?php

namespace MartianooCore\Api\Database;
use MartianooCore\Api\Utils;

/**
 * This class represents a database table row.
 * @package MartianooCore\Api\Database
 */
class Row
{
    /**
     * The table name
     * @var string
     */
    private $tableName = "";

    /**
     * The table's primary key
     * @var string
     */
    private $primaryKey = "";

    private $dbColumns = [];



    /**
     * Creates an instance or throws an exception in the following cases:
     *
     * **$tableName** is not a non-empty string; the exception's message is **Invalid table name**
     * No connection the database was found; the exception's message is **Connection to db failed**
     * @param string $tableName the table name
     * @param array $tableInfos an array containing details about the columns of the table. Each element of this array
     * is an array containing the following keys:
     * * **field** a string representing the column's name
     * * **key** a string indicating if the column is a primary key (value is **pri**) or a unique key (value is **uni**)
     * * **value** a string representing the column's value
     * @throws \Exception
     */
    function __construct($tableName, $tableInfos = [])
    {
        if (! is_string($tableName) || strlen($tableName) == 0) {
            throw new \Exception("Invalid table name");
        }

        $this->tableName = $tableName;
        global $wpdb;

        if (! is_array($tableInfos) || count($tableInfos) == 0) {
            $this->setTableColumnsAsPropertiesFromDescribeQuery($tableName, $wpdb);
        } else {
            $this->setTableColumnsAsPropertiesFromArray($tableInfos);
        }

    }

    /**
     * Inserts or updates a row in a database table.
     * Returns true on success or throws an exception in the following cases:
     * * No connection to the database; the exception message is **Connection to db failed**
     * * An error occured while saving data to the database; the exception message corresponds to the database error message.
     * @return bool
     * @throws \Exception
     */
    public function save()
    {
        global $wpdb;
        if (is_null($wpdb) || ! $wpdb->check_connection()) {
            throw new \Exception("Connection to db failed");
        }

        if (is_null($this->{$this->primaryKey}) || strlen($this->{$this->primaryKey}) == 0) {
            $sql = $this->buildInsertSql($wpdb);
        } else {
            $sql = $this->buildUpdateSql($wpdb);
        }

        $wpdb->hide_errors();
        $wpdb->query($sql);
        if (strlen($wpdb->last_error) > 0) {
            throw new \Exception($wpdb->last_error);
        }

        return true;
    }

    /**
     * Deletes a row in a database table.
     * Returns true on success or throws an exception in the following cases:
     * * No connection to the database; the exception message is **Connection to db failed**
     * * An error occured while saving data to the database; the exception message corresponds to the database error message.
     * @return bool
     * @throws \Exception
     */
    public function delete()
    {
        global $wpdb;
        if (is_null($wpdb) || ! $wpdb->check_connection()) {
            throw new \Exception("Connection to db failed");
        }

        if (strlen($this->{$this->primaryKey}) > 0) {
            $sql = "DELETE FROM " . $this->tableName . " WHERE " . $this->primaryKey . "=%s";
            $sql = $wpdb->prepare($sql, $this->{$this->primaryKey});
            $wpdb->hide_errors();
            $wpdb->query($sql);
            if (strlen($wpdb->last_error) > 0) {
                throw new \Exception($wpdb->last_error);
            }
            return true;
        } else {
            return false;
        }
    }

    /**
     * Returns the table's name.
     * @return string
     */
    public function getTableName()
    {
        return $this->tableName;
    }


    private function setTableColumnsAsPropertiesFromDescribeQuery($tableName, $wpdb)
    {
        $sql = "DESCRIBE " . $tableName;

        if (! is_null($wpdb) && $wpdb->check_connection()) {
            $wpdb->hide_errors();
            $tableInfos = $wpdb->get_results($sql);
            if (strlen($wpdb->last_error) == 0) {
                foreach ($tableInfos as $columnInfos) {
                    $fieldName = $columnInfos->Field;
                    $this->{$fieldName} = "";

                    if (strtolower($columnInfos->Key) == "pri") {
                        $this->primaryKey = $columnInfos->Field;
                    } else {
                        $this->dbColumns["{$fieldName}"] = "";
                    }
                }
            } else {
                throw new \Exception($wpdb->last_error);
            }
        } else {
            throw new \Exception("Connection to db failed");
        }
    }

    private function setTableColumnsAsPropertiesFromArray($tableInfos)
    {
        foreach ($tableInfos as $columnInfos) {
            $fieldName = $columnInfos["field"];
            $this->{$fieldName} = $columnInfos["value"];

            if (strtolower($columnInfos["key"]) == "pri") {
                $this->primaryKey = $columnInfos["field"];
            } else {
                $this->dbColumns["{$fieldName}"] = $columnInfos["value"];
            }
        }
    }


    private function buildInsertSql($wpdb)
    {
        $sql = "INSERT INTO " . $this->tableName . " ";
        $columnsPart = "";
        $valuesPart = "";
        $values = [];
        $this->setCreationDate();
        $this->setModificationDate();
        foreach ($this->dbColumns as $columnName => $value) {
            if (count($values) == 0) {
                $columnsPart .= $columnName;
                $valuesPart .= "%s";
            } else {
                $columnsPart .= "," . $columnName;
                $valuesPart .= ",%s";
            }
            $values[] = $this->{$columnName};
        }
        $sql .= "(" . $columnsPart . ")". " VALUES(" . $valuesPart . ");";
        $prepareSql = $wpdb->prepare($sql, $values);
        return $prepareSql;
    }

    private function buildUpdateSql($wpdb)
    {
        $sql = "UPDATE " . $this->tableName . " SET ";
        $columnsPart = "";
        $wherePart = " WHERE " . $this->primaryKey . "=%s;";

        $values = [];
        $this->setModificationDate();
        foreach ($this->dbColumns as $columnName => $columnValue) {
            if (count($values) == 0) {
                $columnsPart .= $columnName . "=%s";
            } else {
                $columnsPart .= "," . $columnName . "=%s";
            }
            $values[] = $this->{$columnName};
        }
        $values[] = $this->{$this->primaryKey};
        $sql .= $columnsPart . $wherePart;
        $prepareSql = $wpdb->prepare($sql, $values);
        return $prepareSql;
    }

    /**
     * @codeCoverageIgnore
     */
    private function setCreationDate()
    {
        if (property_exists($this, "creation_date")) {
            $this->creation_date = Utils::dateOfDay() . " " . Utils::currentTime();
        }
    }

    /**
     * @codeCoverageIgnore
     */
    private function setModificationDate()
    {
        if (property_exists($this, "modification_date")) {
            $this->modification_date = Utils::dateOfDay() . " " . Utils::currentTime();
        }
    }
}