<?php

namespace MartianooCore\Api\Database;

/**
 * This class represents the schema of a database table.
 * @package MartianooCore\Api\Database
 */
class DbSchema
{
    const MYSQL_STORE_ENGINE_InnoDB = "InnoDB";
    const MYSQL_STORE_ENGINE_MyISAM = "MyISAM";

    /**
     * Returns true if the table was created successfully or throws an exception in the following cases:
     *
     * * No connection the database was found; the exception's message is **Connection to db failed**
     * * An error occurred while creating the table; the exception's message corresponds to the error message.
     * * **$tableName** is not a non-empty string; the exception's message **Invalid table name**
     * @param string $tableName the name of the table to be created
     * @param Spec $columnsSpec The specification of each columns of the table
     * @param string $storageEngine the table's storage engine. Allowed values **MYSQL_STORE_ENGINE_InnoDB**,
     * **MYSQL_STORE_ENGINE_MyISAM**
     * @return bool
     * @throws \Exception
     */
    public static function create($tableName, Spec $columnsSpec, $storageEngine = self::MYSQL_STORE_ENGINE_InnoDB)
    {
        if (! is_string($tableName) || strlen($tableName) == 0) {
            throw new \Exception("Invalid table name");
        }

        $sql = "CREATE TABLE " . $tableName  . "(" . $columnsSpec->toString() . ") ENGINE = " . $storageEngine . ";";
        global $wpdb;
        if (! is_null($wpdb) && $wpdb->check_connection()) {
            $wpdb->hide_errors();
            $wpdb->query($sql);
            if (strlen($wpdb->last_error) > 0) {
                throw new \Exception($wpdb->last_error);
            }
        } else {
            throw new \Exception("Connection to db failed");
        }

        return true;
    }

    /**
     * Returns true if the database table was successfully dropped or throws an exception in the following cases:
     *
     * * No connection the database was found; the exception's message is **Connection to db failed**
     * * An error occurred while dropping the table; the exception's message corresponds to the error message.
     * * **$tableName** is not a non-empty string; the exception's message **Invalid table name**
     * @param string $tableName a table name
     * @return bool
     * @throws \Exception
     */
    public static function drop($tableName)
    {
        if (! is_string($tableName) || strlen($tableName) == 0) {
            throw new \Exception("Invalid table name");
        }

        $sql = "DROP TABLE IF EXISTS " . $tableName . ";";
        global $wpdb;
        if (! is_null($wpdb) && $wpdb->check_connection()) {
            $wpdb->hide_errors();
            $wpdb->query($sql);
            if (strlen($wpdb->last_error) > 0) {
                throw new \Exception($wpdb->last_error);
            }
        } else {
            throw new \Exception("Connection to db failed");
        }

        return true;
    }


    /**
     * Returns true if the database view was successfully created or throws an exception in the following cases:
     * * No connection the database was found; the exception's message is **Connection to db failed**
     * * An error occurred while creating the view; the exception's message corresponds to the error message.
     * * **$viewName** is not valid; the exception's message is **Invalid view name**
     * * **$sqlStmt** is not valid; the exception's message is **Invalid view statement**
     * @param string $viewName a view name
     * @param string $sqlStmt a view SQL statement
     * @return bool
     * @throws \Exception
     */
    public static function createView($viewName, $sqlStmt)
    {
        if (! is_string($viewName) || strlen($viewName) == 0) {
            throw new \Exception("Invalid view name");
        }

        if (! is_string($sqlStmt) || strlen($sqlStmt) == 0) {
            throw new \Exception("Invalid view statement");
        }

        $sql = "CREATE OR REPLACE VIEW " . $viewName . " AS " . $sqlStmt . ";";

        global $wpdb;
        if (! is_null($wpdb) && $wpdb->check_connection()) {
            $wpdb->hide_errors();
            $wpdb->query($sql);
            if (strlen($wpdb->last_error) > 0) {
                throw new \Exception($wpdb->last_error);
            }
        } else {
            throw new \Exception("Connection to db failed");
        }

        return true;
    }

    /**
     * Returns true if the database view was successfully dropped or throws an exception in the following cases:
     *
     * * No connection the database was found; the exception's message is **Connection to db failed**
     * * An error occurred while dropping the view; the exception's message corresponds to the error message.
     * * **$viewName** is not a non-empty string; the exception's message **Invalid view name**
     * @param string $viewName a view name
     * @return bool
     * @throws \Exception
     */
    public static function dropView($viewName)
    {
        if (! is_string($viewName) || strlen($viewName) == 0) {
            throw new \Exception("Invalid view name");
        }

        $sql = "DROP VIEW " . $viewName . ";";

        global $wpdb;
        if (! is_null($wpdb) && $wpdb->check_connection()) {
            $wpdb->hide_errors();
            $wpdb->query($sql);
            if (strlen($wpdb->last_error) > 0) {
                throw new \Exception($wpdb->last_error);
            }
        } else {
            throw new \Exception("Connection to db failed");
        }

        return true;
    }
}