<?php

namespace MartianooCore\Api\Database;


/**
 * This class represents the specification of a database table; i.e. the columns and their data types.
 * @package MartianooCore\Api\Database
 */
class Spec
{
    /**
     * The column's name used as primary key.
     * @var string
     */
    private $primaryKey = "";

    /** The list of table's columns
     * @var array
     */
    private $tableColumns = [];

    private $foreignKeys = [];


    /**
     * Adds a primary key column to a table or throws an exception in the following cases:
     *
     * * **$columnName** is invalid; the exception's message is **Invalid column name**
     * * a primary key is already defined; the exception's message is **Primary key already defined**
     * @param string $column the name of column to be added
     * @throws \Exception
     * @return bool
     */
    public function autoIncrements($column)
    {
        $this->validateName($column);

        if (strlen($this->primaryKey) > 0) {
            throw new \Exception("Primary key already defined");
        }

        $this->primaryKey = $column;
        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "INT UNSIGNED NOT NULL AUTO_INCREMENT",
            "attrib" => new Attrib(),
        ];

        return true;
    }

    /**
     * Adds a foreign key constraint to a table or throws an exception in the following cases:
     *
     * * * **$columnName** is invalid; the exception's message is **Invalid column name**
     * * * **$foreignTableName** is invalid; the exception's message is **Invalid parent table name**
     * * * **$foreignColumnName** is invalid; the exception's message is **Invalid parent column name**
     * @param string $column the child column
     * @param string $parentTable the parent table name
     * @param string $parentColumn the parent colunn
     * @return Attrib
     */
    public function foreignKey($column, $parentTable, $parentColumn)
    {
        $this->validateName($column);
        $this->validateName($parentTable, "Invalid parent table name");
        $this->validateName($parentColumn, "Invalid parent column name");

        $attrib = new Attrib();
        $this->foreignKeys[] = [
            "column" => $column,
            "parent-table" => $parentTable,
            "parent-column" => $parentColumn,
            "attrib" => $attrib,
        ];

        return $attrib;
    }

    /**
     * Adds a column ,with a varchar data type, to the table or throws an exception in the following cases:
     *
     * **$columnName** is invalid; the exception's message is **Invalid column name**
     * **$columnName** is invalid; the exception's message is **Invalid column name**
     * @param string $column the name of column to be added
     * @param integer $length the column's length
     * @throws \Exception
     * @return Attrib
     */
    public function string($column, $length)
    {
        $this->validateName($column);

        if (! is_int($length) || $length <= 0) {
            throw new \Exception("Invalid length");
        }

        $attrib = new Attrib();
        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "VARCHAR(" . $length . ")",
            "attrib" => $attrib
        ];

        return $attrib;
    }

    /**
     * Adds a column ,with a integer data type, to the table or throws an exception.
     * The exception's message is **Invalid column name**
     * @param string $column the name of column to be added
     * @throws \Exception
     * @return Attrib
     */
    public function integer($column)
    {
        $this->validateName($column);

        $attrib = new Attrib();
        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "INTEGER",
            "attrib" => $attrib
        ];

        return $attrib;
    }

    /**
     * Adds a column ,with a text data type, to the table or throws an exception.
     * The exception's message is **Invalid column name**
     * @param string $column the name of column to be added
     * @return Attrib
     */
    public function text($column)
    {
        $this->validateName($column);

        $attrib = new Attrib();
        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "TEXT",
            "attrib" => $attrib
        ];

        return $attrib;
    }

    /**
     * Adds a column ,with a date data type, to the table or throws an exception.
     * The exception's message is **Invalid column name**
     * @param string $column the name of column to be added
     * @return Attrib
     */
    public function date($column)
    {
        $this->validateName($column);

        $attrib = new Attrib();
        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "DATE",
            "attrib" => $attrib
        ];

        return $attrib;
    }

    /**
     * Adds a column ,with a datetime data type, to the table or throws an exception.
     * The exception's message is **Invalid column name**
     * @param string $column the name of column to be added
     * @return Attrib
     */
    public function datetime($column)
    {
        $this->validateName($column);

        $attrib = new Attrib();
        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "DATETIME",
            "attrib" => $attrib
        ];

        return $attrib;
    }


    /**
     * Adds a column ,with an ENUM data type, to the table or throws an exception in the following cases:
     *
     * * **$columnName** is not a non-empty string, the exception's message is **Invalid column name**
     * * **$allowedValues** is not a non-empty array, the exception's message is **Invalid allowed values*
     * @param string $column the name of column to be added
     * @param array $allowedValues
     * @return Attrib
     * @throws \Exception
     */
    public function enum($column, $allowedValues)
    {
        $this->validateName($column);

        if (! is_array($allowedValues) || count($allowedValues) == 0) {
            throw new \Exception("Invalid allowed values");
        }

        $valuesPart = "";
        foreach ($allowedValues as $value) {
            if (strlen($valuesPart) == 0) {
                $valuesPart .= "'" . $value . "'";
            } else {
                $valuesPart .= ",'" . $value . "'";
            }
        }

        $attrib = new Attrib();

        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "ENUM(" . $valuesPart . ")",
            "attrib" => $attrib
        ];

        return $attrib;
    }


    /**
     * Adds a column ,with a DECIMAL data type, to the table or throws an exception in the following cases:
     *
     * * **$columnName** is not a non-empty string, the exception's message is **Invalid column name**
     * * **$precision** is not a positive integer, the exception's message is **Invalid precision**
     * * **$decimals** is not a positive integer, the exception's message is **Invalid decimals**
     * @param string $column the column name
     * @param integer $precision the number of digits before the decimal point.
     * @param integer $decimals the number of decimals
     * @return Attrib
     * @throws \Exception
     *
     */
    public function decimal($column, $precision, $decimals)
    {
        $this->validateName($column);

        if (! is_int($precision) || $precision <= 0) {
            throw new \Exception("Invalid precision");
        }

        if (! is_int($decimals) || $decimals <= 0) {
            throw new \Exception("Invalid decimals");
        }

        $attrib = new Attrib();

        $this->tableColumns[] = [
            "name" => $column,
            "data-type" => "DECIMAL(" . $precision . "," . $decimals . ")",
            "attrib" => $attrib
        ];

        return $attrib;
    }

    /**
     * Adds the default creation_date and modification_date from a table's list of columns. These columns are populated
     * automatically on insertion and update of the table's data.
     */
    public function defaultDates()
    {
        $this->tableColumns[] = [
            "name" => "creation_date",
            "data-type" => "DATETIME",
            "attrib" => new Attrib(),
        ];

        $this->tableColumns[] = [
            "name" => "modification_date",
            "data-type" => "DATETIME",
            "attrib" => new Attrib(),
        ];
    }

    /**
     * Returns the a comma-separated list of table's columns or in case no column was added it throws an exception.
     * The exception's message is **Invalid spec**.
     *
     * ##### Example:
     * ```
     * $spec = new Spec();
     * $spec->autoIncrements("col1"); // Add primary key
     * $spec->string("col2", 30)->notNull(); // add a varchar column
     * $spec->text("col3"); // add a text column
     * $spec->integer("col4"); // add an integer column
     * echo $spec->toString();
     * ```
     * <p>
     * The code above will output:
     * </p>
     * ```
     * col1 INT UNSIGNED NOT NULL AUTO_INCREMENT,
     * col2 VARCHAR(30) NOT NULL,
     * col3 TEXT,
     * col4 INTEGER,
     * PRIMARY KEY (col1)
     * ```
     * @return string
     * @throws \Exception
     */
    public function toString()
    {
        if (count($this->tableColumns) == 0) {
            throw new \Exception("Invalid spec");
        }

        $specText = "";

        foreach ($this->tableColumns as $column) {
            if (strlen($specText) == 0) {
                $specText .= $column["name"] . " " . $column["data-type"];
                $specText .= $column["attrib"]->toString();
            } else {
                $specText .= "," .$column["name"] . " " . $column["data-type"];
                $specText .= $column["attrib"]->toString();
            }
        }

        if (strlen($this->primaryKey) > 0) {
            $specText .= ",PRIMARY KEY (" . $this->primaryKey . ")";
        }

        foreach ($this->foreignKeys as $foreignKey) {
            $specText .= ",FOREIGN KEY (" . $foreignKey["column"] . ")";
            $specText .= " REFERENCES " . $foreignKey["parent-table"] . "(" . $foreignKey["parent-column"] . ")";
            $specText .= $foreignKey["attrib"]->toString();
        }

        return $specText;
    }


    /**
     * @param $name
     * @throws \Exception
     * @codeCoverageIgnore
     */
    private function validateName($name, $message = "")
    {
        if (! is_string($name) || strlen($name) == 0) {
            if (strlen($message) > 0) {
                throw new \Exception($message);
            }
            throw new \Exception("Invalid column name");
        }
    }
}